<?php
/**
 * HRM UI Helper Functions
 * Utility functions for rendering enhanced UI components
 *
 * @package HR_Management
 * @since 2.0.0
 */

defined( 'ABSPATH' ) || exit;

class HRM_UI_Helper {

    /**
     * Render an enhanced table
     *
     * @param array $args {
     *     @type array  $columns  Column headers
     *     @type array  $rows     Table rows data
     *     @type string $class    Additional CSS classes
     *     @type bool   $striped  Enable striped rows
     *     @type bool   $compact  Use compact padding
     * }
     */
    public static function render_table( $args = array() ) {
        $defaults = array(
            'columns' => array(),
            'rows'    => array(),
            'class'   => '',
            'striped' => false,
            'compact' => false,
        );

        $args = wp_parse_args( $args, $defaults );
        
        $table_class = 'hrm-table';
        if ( $args['striped'] ) {
            $table_class .= ' striped';
        }
        if ( $args['compact'] ) {
            $table_class .= ' compact';
        }
        if ( $args['class'] ) {
            $table_class .= ' ' . esc_attr( $args['class'] );
        }

        ?>
        <div class="hrm-table-wrapper">
            <table class="<?php echo esc_attr( $table_class ); ?>">
                <thead>
                    <tr>
                        <?php foreach ( $args['columns'] as $column ) : ?>
                            <th><?php echo esc_html( $column ); ?></th>
                        <?php endforeach; ?>
                    </tr>
                </thead>
                <tbody>
                    <?php if ( !empty( $args['rows'] ) ) : ?>
                        <?php foreach ( $args['rows'] as $row ) : ?>
                            <tr>
                                <?php foreach ( $row as $cell ) : ?>
                                    <td><?php echo $cell; // Should be pre-escaped ?></td>
                                <?php endforeach; ?>
                            </tr>
                        <?php endforeach; ?>
                    <?php else : ?>
                        <tr>
                            <td colspan="<?php echo count( $args['columns'] ); ?>" style="text-align: center; padding: 2rem;">
                                <?php self::render_empty_state( array(
                                    'title'       => __( 'No data found', 'hr-management' ),
                                    'description' => __( 'There are no records to display.', 'hr-management' ),
                                ) ); ?>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    /**
     * Render a badge
     *
     * @param string $text  Badge text
     * @param string $type  Badge type (success, warning, danger, info, neutral, primary)
     * @param bool   $dot   Show dot indicator
     * @return string       HTML output
     */
    public static function render_badge( $text, $type = 'neutral', $dot = false ) {
        $class = 'hrm-badge hrm-badge-' . esc_attr( $type );
        if ( $dot ) {
            $class .= ' hrm-badge-dot';
        }

        return sprintf(
            '<span class="%s">%s</span>',
            $class,
            esc_html( $text )
        );
    }

    /**
     * Render status badge based on status text
     *
     * @param string $status Status text
     * @return string        HTML output
     */
    public static function render_status_badge( $status ) {
        $status_lower = strtolower( trim( $status ) );
        
        // Map status to badge type
        $type_map = array(
            'present'     => 'success',
            'active'      => 'success',
            'completed'   => 'success',
            'approved'    => 'success',
            'success'     => 'success',
            
            'absent'      => 'danger',
            'inactive'    => 'danger',
            'rejected'    => 'danger',
            'failed'      => 'danger',
            'error'       => 'danger',
            
            'pending'     => 'warning',
            'in-progress' => 'warning',
            'in progress' => 'warning',
            'processing'  => 'warning',
            'warning'     => 'warning',
            
            'info'        => 'info',
            'draft'       => 'info',
        );

        $type = isset( $type_map[ $status_lower ] ) ? $type_map[ $status_lower ] : 'neutral';
        
        return self::render_badge( $status, $type, true );
    }

    /**
     * Render a button
     *
     * @param array $args {
     *     @type string $text     Button text
     *     @type string $type     Button type (primary, outline, danger, red)
     *     @type string $size     Button size (normal, sm, lg)
     *     @type string $url      Button URL (for links)
     *     @type string $icon     Dashicon class
     *     @type array  $attrs    Additional HTML attributes
     *     @type string $onclick  JavaScript onclick handler
     * }
     * @return string  HTML output
     */
    public static function render_button( $args = array() ) {
        $defaults = array(
            'text'    => '',
            'type'    => 'primary',
            'size'    => '',
            'url'     => '',
            'icon'    => '',
            'attrs'   => array(),
            'onclick' => '',
        );

        $args = wp_parse_args( $args, $defaults );

        $class = 'hrm-btn-' . esc_attr( $args['type'] );
        if ( $args['size'] ) {
            $class .= ' btn-' . esc_attr( $args['size'] );
        }

        $attributes = '';
        if ( $args['onclick'] ) {
            $attributes .= ' onclick="' . esc_js( $args['onclick'] ) . '"';
        }

        foreach ( $args['attrs'] as $key => $value ) {
            $attributes .= ' ' . esc_attr( $key ) . '="' . esc_attr( $value ) . '"';
        }

        $icon_html = '';
        if ( $args['icon'] ) {
            // Support both dashicons format and new HRM icon format
            if ( strpos( $args['icon'], 'dashicons-' ) === 0 ) {
                $icon_html = '<span class="dashicons ' . esc_attr( $args['icon'] ) . '"></span>';
            } elseif ( class_exists( 'HRM_Icons' ) ) {
                // Use new icon system
                $icon_html = HRM_Icons::get( $args['icon'], ['size' => 'sm'] );
            } else {
                $icon_html = '<span class="dashicons ' . esc_attr( $args['icon'] ) . '"></span>';
            }
        }

        if ( $args['url'] ) {
            return sprintf(
                '<a href="%s" class="%s"%s>%s%s</a>',
                esc_url( $args['url'] ),
                $class,
                $attributes,
                $icon_html,
                esc_html( $args['text'] )
            );
        } else {
            return sprintf(
                '<button class="%s"%s>%s%s</button>',
                $class,
                $attributes,
                $icon_html,
                esc_html( $args['text'] )
            );
        }
    }

    /**
     * Render a card
     *
     * @param array $args {
     *     @type string $title   Card title
     *     @type string $content Card body content
     *     @type string $footer  Card footer content
     *     @type string $class   Additional CSS classes
     * }
     */
    public static function render_card( $args = array() ) {
        $defaults = array(
            'title'   => '',
            'content' => '',
            'footer'  => '',
            'class'   => '',
        );

        $args = wp_parse_args( $args, $defaults );

        $class = 'hrm-card-enhanced';
        if ( $args['class'] ) {
            $class .= ' ' . esc_attr( $args['class'] );
        }

        ?>
        <div class="<?php echo esc_attr( $class ); ?>">
            <?php if ( $args['title'] ) : ?>
                <div class="hrm-card-header-enhanced">
                    <h3><?php echo esc_html( $args['title'] ); ?></h3>
                </div>
            <?php endif; ?>
            
            <?php if ( $args['content'] ) : ?>
                <div class="hrm-card-body-enhanced">
                    <?php echo $args['content']; // Should be pre-escaped ?>
                </div>
            <?php endif; ?>
            
            <?php if ( $args['footer'] ) : ?>
                <div class="hrm-card-footer-enhanced">
                    <?php echo $args['footer']; // Should be pre-escaped ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render filter bar
     *
     * @param array $args {
     *     @type array  $filters Filters configuration
     *     @type array  $actions Action buttons configuration
     * }
     */
    public static function render_filter_bar( $args = array() ) {
        $defaults = array(
            'filters' => array(),
            'actions' => array(),
        );

        $args = wp_parse_args( $args, $defaults );

        ?>
        <div class="hrm-filter-bar-enhanced">
            <div class="hrm-filter-group">
                <?php foreach ( $args['filters'] as $filter ) : ?>
                    <?php if ( $filter['type'] === 'search' ) : ?>
                        <div class="hrm-search-input">
                            <input 
                                type="text" 
                                class="hrm-input" 
                                name="<?php echo esc_attr( $filter['name'] ); ?>"
                                placeholder="<?php echo esc_attr( $filter['placeholder'] ?? __( 'Search...', 'hr-management' ) ); ?>"
                            >
                        </div>
                    <?php elseif ( $filter['type'] === 'select' ) : ?>
                        <select class="hrm-select" name="<?php echo esc_attr( $filter['name'] ); ?>">
                            <?php foreach ( $filter['options'] as $value => $label ) : ?>
                                <option value="<?php echo esc_attr( $value ); ?>">
                                    <?php echo esc_html( $label ); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    <?php elseif ( $filter['type'] === 'date' ) : ?>
                        <input 
                            type="date" 
                            class="hrm-input" 
                            name="<?php echo esc_attr( $filter['name'] ); ?>"
                        >
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
            
            <?php if ( !empty( $args['actions'] ) ) : ?>
                <div class="hrm-filter-group">
                    <?php foreach ( $args['actions'] as $action ) : ?>
                        <?php echo self::render_button( $action ); ?>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render pagination
     *
     * @param int    $current_page Current page number
     * @param int    $total_pages  Total number of pages
     * @param string $base_url     Base URL for pagination links
     */
    public static function render_pagination( $current_page, $total_pages, $base_url = '' ) {
        if ( $total_pages <= 1 ) {
            return;
        }

        $page_links = array();
        
        // Previous button
        $page_links[] = sprintf(
            '<button class="hrm-pagination-btn" %s onclick="changePage(%d)"><span class="dashicons dashicons-arrow-left-alt2"></span> Previous</button>',
            $current_page <= 1 ? 'disabled' : '',
            max( 1, $current_page - 1 )
        );

        // Page numbers
        $range = 2; // Number of pages to show on each side of current page
        
        for ( $i = 1; $i <= $total_pages; $i++ ) {
            if ( $i == 1 || $i == $total_pages || ( $i >= $current_page - $range && $i <= $current_page + $range ) ) {
                $active_class = ( $i == $current_page ) ? ' active' : '';
                $page_links[] = sprintf(
                    '<button class="hrm-pagination-btn%s" onclick="changePage(%d)">%d</button>',
                    $active_class,
                    $i,
                    $i
                );
            } elseif ( $i == $current_page - $range - 1 || $i == $current_page + $range + 1 ) {
                $page_links[] = '<span class="hrm-pagination-ellipsis">...</span>';
            }
        }

        // Next button
        $page_links[] = sprintf(
            '<button class="hrm-pagination-btn" %s onclick="changePage(%d)">Next <span class="dashicons dashicons-arrow-right-alt2"></span></button>',
            $current_page >= $total_pages ? 'disabled' : '',
            min( $total_pages, $current_page + 1 )
        );

        ?>
        <div class="hrm-pagination">
            <div class="hrm-pagination-info">
                <?php printf(
                    esc_html__( 'Page %d of %d', 'hr-management' ),
                    $current_page,
                    $total_pages
                ); ?>
            </div>
            <div class="hrm-pagination-controls">
                <?php echo implode( '', $page_links ); ?>
            </div>
        </div>
        <?php
    }

    /**
     * Render empty state
     *
     * @param array $args {
     *     @type string $title       Empty state title
     *     @type string $description Empty state description
     *     @type string $icon        Dashicon class
     *     @type array  $action      Action button configuration
     * }
     */
    public static function render_empty_state( $args = array() ) {
        $defaults = array(
            'title'       => __( 'No items found', 'hr-management' ),
            'description' => __( 'There are no items to display.', 'hr-management' ),
            'icon'        => 'dashicons-info',
            'action'      => null,
        );

        $args = wp_parse_args( $args, $defaults );

        ?>
        <div class="hrm-empty-state">
            <?php if ( $args['icon'] ) : ?>
                <div class="hrm-empty-state-icon">
                    <span class="dashicons <?php echo esc_attr( $args['icon'] ); ?>"></span>
                </div>
            <?php endif; ?>
            
            <h3 class="hrm-empty-state-title"><?php echo esc_html( $args['title'] ); ?></h3>
            
            <?php if ( $args['description'] ) : ?>
                <p class="hrm-empty-state-description"><?php echo esc_html( $args['description'] ); ?></p>
            <?php endif; ?>
            
            <?php if ( $args['action'] ) : ?>
                <div class="hrm-empty-state-action">
                    <?php echo self::render_button( $args['action'] ); ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render progress bar
     *
     * @param int    $percentage Progress percentage (0-100)
     * @param string $type       Progress type (primary, success, warning, danger)
     * @param string $label      Progress label
     */
    public static function render_progress_bar( $percentage, $type = 'primary', $label = '' ) {
        $percentage = max( 0, min( 100, intval( $percentage ) ) );
        $class = $type !== 'primary' ? ' ' . esc_attr( $type ) : '';

        ?>
        <div class="hrm-progress-wrapper">
            <?php if ( $label ) : ?>
                <div class="hrm-progress-label">
                    <span><?php echo esc_html( $label ); ?></span>
                    <span><?php echo esc_html( $percentage ); ?>%</span>
                </div>
            <?php endif; ?>
            <div class="hrm-progress">
                <div class="hrm-progress-bar<?php echo $class; ?>" style="width: <?php echo esc_attr( $percentage ); ?>%;"></div>
            </div>
        </div>
        <?php
    }

    /**
     * Render alert
     *
     * @param string $message Alert message
     * @param string $type    Alert type (success, warning, danger, info)
     * @param string $title   Alert title (optional)
     * @param bool   $dismissible Whether alert can be dismissed
     */
    public static function render_alert( $message, $type = 'info', $title = '', $dismissible = true ) {
        $icon_map = array(
            'success' => 'dashicons-yes-alt',
            'warning' => 'dashicons-warning',
            'danger'  => 'dashicons-dismiss',
            'error'   => 'dashicons-dismiss',
            'info'    => 'dashicons-info',
        );

        $icon = isset( $icon_map[ $type ] ) ? $icon_map[ $type ] : 'dashicons-info';

        ?>
        <div class="hrm-alert hrm-alert-<?php echo esc_attr( $type ); ?>">
            <span class="dashicons <?php echo esc_attr( $icon ); ?> hrm-alert-icon"></span>
            <div class="hrm-alert-content">
                <?php if ( $title ) : ?>
                    <div class="hrm-alert-title"><?php echo esc_html( $title ); ?></div>
                <?php endif; ?>
                <div class="hrm-alert-message"><?php echo esc_html( $message ); ?></div>
            </div>
            <?php if ( $dismissible ) : ?>
                <button class="hrm-alert-close" onclick="this.parentElement.style.display='none'">&times;</button>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Get inline loading spinner
     *
     * @param string $size Spinner size (sm, md, lg)
     * @return string      HTML output
     */
    public static function get_spinner( $size = 'md' ) {
        $size_map = array(
            'sm' => '16px',
            'md' => '24px',
            'lg' => '40px',
        );

        $dimension = isset( $size_map[ $size ] ) ? $size_map[ $size ] : $size_map['md'];

        return sprintf(
            '<div class="hrm-loading-spinner" style="width: %s; height: %s; border-width: 3px;"></div>',
            esc_attr( $dimension ),
            esc_attr( $dimension )
        );
    }

    /**
     * Render avatar
     *
     * @param string $name    Name for initials
     * @param string $url     Image URL (optional)
     * @param string $size    Avatar size (sm, md, lg)
     * @param string $tooltip Tooltip text
     * @return string         HTML output
     */
    public static function render_avatar( $name, $url = '', $size = 'md', $tooltip = '' ) {
        $class = 'hrm-avatar';
        if ( $size !== 'md' ) {
            $class .= ' hrm-avatar-' . esc_attr( $size );
        }

        $initials = self::get_initials( $name );
        
        $tooltip_attr = $tooltip ? ' data-tooltip="' . esc_attr( $tooltip ) . '"' : '';

        if ( $url ) {
            return sprintf(
                '<div class="%s"%s><img src="%s" alt="%s"></div>',
                $class,
                $tooltip_attr,
                esc_url( $url ),
                esc_attr( $name )
            );
        } else {
            return sprintf(
                '<div class="%s"%s>%s</div>',
                $class,
                $tooltip_attr,
                esc_html( $initials )
            );
        }
    }

    /**
     * Get initials from name
     *
     * @param string $name Full name
     * @return string      Initials
     */
    private static function get_initials( $name ) {
        $words = explode( ' ', trim( $name ) );
        $initials = '';
        
        foreach ( array_slice( $words, 0, 2 ) as $word ) {
            $initials .= strtoupper( substr( $word, 0, 1 ) );
        }
        
        return $initials ?: 'U';
    }
}
